var __extends = this.__extends || function (d, b) {
    function __() { this.constructor = d; }
    __.prototype = b.prototype;
    d.prototype = new __();
};
var VisualProfiler;
(function (VisualProfiler) {
    var DataSeriesInfo = (function () {
        function DataSeriesInfo(label, name, cssClass, sortOrder) {
            if(!name || sortOrder === undefined || sortOrder === null) {
                throw new Error(Plugin.Resources.getErrorString("JSPerf.1044"));
            }
            this._label = label;
            this._name = name;
            this._cssClass = cssClass;
            this._sortOrder = sortOrder;
        }
        Object.defineProperty(DataSeriesInfo.prototype, "cssClass", {
            get: function () {
                return this._cssClass;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(DataSeriesInfo.prototype, "label", {
            get: function () {
                return this._label;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(DataSeriesInfo.prototype, "name", {
            get: function () {
                return this._name;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(DataSeriesInfo.prototype, "sortOrder", {
            get: function () {
                return this._sortOrder;
            },
            enumerable: true,
            configurable: true
        });
        return DataSeriesInfo;
    })();
    VisualProfiler.DataSeriesInfo = DataSeriesInfo;    
    var StackedBarChartPresenter = (function () {
        function StackedBarChartPresenter(options) {
            this._data = new Array();
            this._dataSeriesInfo = {
            };
            this._maximumYValue = Number.NEGATIVE_INFINITY;
            this._yAxisValuePadding = 10;
            this.viewModel = [];
            this._options = options;
            this.validateOptions();
            this._pixelHorizontalValue = this.xWidth / (this._options.width - this._options.yAxisWidth - this._options.rightPadding);
        }
        Object.defineProperty(StackedBarChartPresenter.prototype, "dataSelectionChanged", {
            set: function (dataSelectionChangedCallback) {
                this._dataSelectionChanged = dataSelectionChangedCallback;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(StackedBarChartPresenter.prototype, "selection", {
            set: function (selection) {
                if(!selection) {
                    return;
                }
                var startXPercent = (selection.begin.nsec - this._options.minX) / this.xWidth * 100;
                var widthPercent = selection.elapsed.nsec / this.xWidth * 100;
                if(this._dataSelectionChanged) {
                    this._dataSelectionChanged(startXPercent, widthPercent);
                }
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(StackedBarChartPresenter.prototype, "xWidth", {
            get: function () {
                return this._options.maxX - this._options.minX;
            },
            enumerable: true,
            configurable: true
        });
        StackedBarChartPresenter.prototype.addData = function (chartData) {
            var _this = this;
            chartData.forEach(function (dataItem) {
                if(_this._dataSeriesInfo.hasOwnProperty(dataItem.series)) {
                    _this._data.push(dataItem);
                } else {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1043"));
                }
            });
            this.generateViewModel();
        };
        StackedBarChartPresenter.prototype.addSeries = function (seriesInfo) {
            for(var i = 0; i < seriesInfo.length; i++) {
                var info = seriesInfo[i];
                if(this._dataSeriesInfo.hasOwnProperty(info.name)) {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1045"));
                }
                this._dataSeriesInfo[info.name] = info;
            }
        };
        StackedBarChartPresenter.prototype.getViewOptions = function () {
            var yAxisMarks = this.determineYAxisMarkPositions(this._options.yAxisMarkValues);
            var viewOptions = {
                ariaDescription: this._options.ariaDescription,
                ariaLabelCallback: this._options.ariaLabelCallback,
                height: this._options.height,
                width: this._options.width,
                title: this._options.title,
                yAxisMarks: yAxisMarks,
                toolTipCallback: this._options.toolTipCallback,
                displayLegend: this._options.displayLegend,
                legendData: this._dataSeriesInfo,
                minSelectionWidthInPixels: this._options.minSelectionWidthInPixels,
                selectionCallback: this.viewSelectionChanged.bind(this),
                yAxisWidth: this._options.yAxisWidth,
                rightPadding: this._options.rightPadding
            };
            return viewOptions;
        };
        StackedBarChartPresenter.prototype.convertChartAreaPercentToDataValue = function (percent) {
            return Math.round(percent * this.xWidth / 100) + this._options.minX;
        };
        StackedBarChartPresenter.prototype.determineYAxisMarkPositions = function (yAxisMarks) {
            var result = {
            };
            for(var i = 0; i < yAxisMarks.length; i++) {
                if(yAxisMarks[i] > this._maximumYValue) {
                    break;
                }
                var yPixelPosition = this._options.height - (yAxisMarks[i] / this._pixelVerticalValue);
                result[yAxisMarks[i]] = yPixelPosition;
            }
            return result;
        };
        StackedBarChartPresenter.prototype.determineYAxisScale = function (allBars) {
            for(var i = 0; i < allBars.length; i++) {
                var totalStackHeight = 0;
                var currentBar = allBars[i];
                for(var j = 0; j < currentBar.length; j++) {
                    var stackComponent = currentBar[j];
                    if(stackComponent.height > 0) {
                        totalStackHeight += stackComponent.height;
                    }
                }
                this._maximumYValue = Math.max(this._maximumYValue, totalStackHeight);
            }
            this._maximumYValue = Math.max(this._options.minYHeight, this._maximumYValue);
            this._maximumYValue += this._yAxisValuePadding;
            this._pixelVerticalValue = Math.ceil(this._maximumYValue - this._options.minY) / this._options.height;
        };
        StackedBarChartPresenter.prototype.generateViewModel = function () {
            var allBars = [
                []
            ];
            var singleBar = [];
            var barWidthAndMargin = this._options.barWidth + this._options.barGap;
            var currentXValue = this._options.minX;
            var prevValue = Number.NEGATIVE_INFINITY;
            var x = 0;
            var i = 0;
            while(i < this._data.length) {
                var dataItem = this._data[i];
                if(dataItem.x < prevValue) {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1046"));
                }
                if(dataItem.x > this._options.maxX) {
                    break;
                }
                prevValue = dataItem.x;
                var currentXValue = Math.floor(x * this._pixelHorizontalValue + this._options.minX);
                var currentBarMinValue = currentXValue;
                var currentBarMaxValue = currentXValue + Math.floor((this._options.barWidth + this._options.barGap) * this._pixelHorizontalValue);
                if(dataItem.x < currentBarMinValue) {
                    i++;
                    continue;
                }
                if(dataItem.x < currentBarMaxValue) {
                    dataItem.x = x;
                    singleBar.push(dataItem);
                    i++;
                } else {
                    allBars.push(singleBar);
                    singleBar = [];
                    x += barWidthAndMargin;
                }
            }
            allBars.push(singleBar);
            this.determineYAxisScale(allBars);
            for(var i = 0; i < allBars.length; i++) {
                this.generateViewModelForSingleStack(allBars[i]);
            }
        };
        StackedBarChartPresenter.prototype.generateViewModelForSingleStack = function (dataItems) {
            if(!dataItems || dataItems.length === 0) {
                return;
            }
            dataItems.sort(this.sortBySeries.bind(this));
            var accumulatedHeight = 0;
            var maxHeightExceeded = false;
            var singleBarViewModel = [];
            for(var i = dataItems.length - 1; i >= 0; i--) {
                var dataItem = dataItems[i];
                if(dataItem.height <= 0) {
                    continue;
                }
                var barHeight = Math.ceil(dataItem.height / this._pixelVerticalValue);
                var startY = this._options.height - (barHeight + accumulatedHeight);
                if(startY < 0) {
                    barHeight = this._options.height - accumulatedHeight;
                    startY = 0;
                    maxHeightExceeded = true;
                }
                accumulatedHeight += barHeight;
                if(this._options.showStackGap && barHeight > 1) {
                    barHeight -= 1;
                    startY += 1;
                }
                var rectangle = {
                    x: dataItem.x,
                    y: startY,
                    height: barHeight,
                    width: this._options.barWidth,
                    className: this._dataSeriesInfo[dataItem.series].cssClass,
                    attachedObject: dataItem.attachedObject
                };
                this.viewModel.push(rectangle);
                if(maxHeightExceeded) {
                    break;
                }
            }
        };
        StackedBarChartPresenter.prototype.sortBySeries = function (chartItem1, chartItem2) {
            return this._dataSeriesInfo[chartItem2.series].sortOrder - this._dataSeriesInfo[chartItem1.series].sortOrder;
        };
        StackedBarChartPresenter.prototype.validateOptions = function () {
            if(!this._options) {
                throw new Error(Plugin.Resources.getErrorString("JSPerf.1047"));
            }
            if((this._options.minX === undefined || this._options.minX === null) || (this._options.maxX === undefined || this._options.maxX === null) || (this._options.minY === undefined || this._options.minY === null) || (this._options.minX > this._options.maxX) || (!this._options.height || !this._options.width || this._options.height < 0 || this._options.width < 0) || (!this._options.barWidth || this._options.barWidth < 0)) {
                throw new Error(Plugin.Resources.getErrorString("JSPerf.1048"));
            }
            this._options.barGap = this._options.barGap || 0;
            this._options.showStackGap = this._options.showStackGap || false;
            this._options.yAxisWidth = this._options.yAxisWidth || 0;
            this._options.minYHeight = this._options.minYHeight || this._options.minY;
        };
        StackedBarChartPresenter.prototype.viewSelectionChanged = function (startPercent, widthPercent) {
            if(!this._options.selectionCallback) {
                return;
            }
            var min = this.convertChartAreaPercentToDataValue(startPercent);
            var max = this.convertChartAreaPercentToDataValue(startPercent + widthPercent);
            this._options.selectionCallback(min, max);
        };
        return StackedBarChartPresenter;
    })();
    VisualProfiler.StackedBarChartPresenter = StackedBarChartPresenter;    
    var StackedBarChartView = (function (_super) {
        __extends(StackedBarChartView, _super);
        function StackedBarChartView() {
                _super.call(this);
            this._idCount = 0;
            this._isCollapsed = false;
            this._selectedId = -1;
            this._selectionStartXPercent = 0;
            this._selectionWidthPercent = 0;
            this._selectionStartXPercent = 0;
            this._selectionWidthPercent = 100;
        }
        StackedBarChartView._barIdPrefix = "bar";
        Object.defineProperty(StackedBarChartView.prototype, "presenter", {
            set: function (presenter) {
                this.unregisterPresenterEvents();
                this._presenter = presenter;
                this.registerPresenterEvents();
                this._viewData = this._presenter.viewModel;
                this._options = presenter.getViewOptions();
                this._barGraphWidth = this._options.width - this._options.yAxisWidth - this._options.rightPadding;
                this.drawChart();
            },
            enumerable: true,
            configurable: true
        });
        StackedBarChartView.prototype.addYAxisMark = function (yAxis, value, cssClass) {
            var mark = document.createElement("div");
            mark.innerText = value;
            var yPos = this._options.yAxisMarks[value];
            mark.classList.add(cssClass);
            yAxis.appendChild(mark);
            yPos -= mark.offsetHeight / 2;
            yPos = Math.max(yPos, 0);
            if((yPos + mark.offsetHeight) >= this._options.height) {
                yPos = this._options.height - mark.offsetHeight;
            }
            mark.style.top = yPos + "px";
        };
        StackedBarChartView.prototype.clearAll = function () {
            this.rootElement.innerHTML = "";
            this._leftUnselectedRegion = null;
            this._rightUnselectedRegion = null;
        };
        StackedBarChartView.prototype.convertPageXToChartAreaPercent = function (pageX) {
            return (pageX - (this._chartAreaContainer.offsetLeft + this._options.yAxisWidth)) / this._barGraphWidth * 100;
        };
        StackedBarChartView.prototype.createContainer = function () {
            this._chartAreaContainer = document.createElement("div");
            this._chartAreaContainer.style.width = this._options.width + "px";
            this._chartAreaContainer.style.height = this._options.height + "px";
            this._chartAreaContainer.classList.add("stackedBarChart");
            this._chartAreaContainer.style.display = this._isCollapsed ? "none" : "-ms-grid";
            this.rootElement.appendChild(this._chartAreaContainer);
        };
        StackedBarChartView.prototype.createLegendItem = function (name, className) {
            var legendItem = document.createElement("div");
            legendItem.classList.add("legendItem");
            var legendColorDiv = document.createElement("div");
            legendColorDiv.classList.add("legendColor");
            legendColorDiv.classList.add(className);
            var legendTextSpan = document.createElement("div");
            legendTextSpan.classList.add("legendText");
            legendTextSpan.innerHTML = name;
            legendItem.appendChild(legendColorDiv);
            legendItem.appendChild(legendTextSpan);
            return legendItem;
        };
        StackedBarChartView.prototype.createYAxisLine = function (axisValue) {
            var line = document.createElement("div");
            line.classList.add("yAxisLine");
            line.style.top = axisValue + "px";
            return line;
        };
        StackedBarChartView.prototype.createRect = function (x, y, height, width, className) {
            var rect = document.createElement("div");
            rect.id = StackedBarChartView._barIdPrefix + this._idCount;
            rect.tabIndex = -1;
            this._idCount++;
            rect.classList.add(className);
            rect.style.position = "absolute";
            rect.style.left = x + "px";
            rect.style.bottom = (this._options.height - y - height) + "px";
            rect.style.height = height + "px";
            rect.style.width = width + "px";
            return rect;
        };
        StackedBarChartView.prototype.createUnselectedRegion = function (xPercent, widthPercent) {
            var rect = document.createElement("div");
            rect.style.height = this._barGraph.offsetHeight + "px";
            rect.style.width = widthPercent + "%";
            rect.style.left = xPercent + "%";
            rect.style.top = "0px";
            rect.className = "unselected";
            return rect;
        };
        StackedBarChartView.prototype.createVerticalRulerLine = function (xPercent) {
            var x = this._barGraphWidth * xPercent / 100;
            var line = document.createElement("div");
            line.style.left = x + "px";
            line.style.height = "100%";
            line.style.top = "0px";
            line.className = "verticalRulerLine";
            return line;
        };
        StackedBarChartView.prototype.createYAxisDiv = function (cssClass) {
            var yAxis = document.createElement("div");
            yAxis.classList.add(cssClass);
            yAxis.style.height = this._options.height + "px";
            yAxis.style.width = this._options.yAxisWidth + "px";
            this._chartAreaContainer.appendChild(yAxis);
            var finalWidthWithStyling = yAxis.offsetWidth;
            if(finalWidthWithStyling !== this._options.yAxisWidth) {
                yAxis.style.width = this._options.yAxisWidth - (finalWidthWithStyling - this._options.yAxisWidth) + "px";
            }
            return yAxis;
        };
        StackedBarChartView.prototype.drag = function (event) {
            var xPercent = this.convertPageXToChartAreaPercent(event.pageX);
            this._dragDirection = StackedBarChartView.getDragDirection(xPercent - this._initialSelectionStartXPercent);
            this._selectionWidthPercent = Math.abs(xPercent - this._initialSelectionStartXPercent);
            if(this._dragDirection === VisualProfiler.DragDirection.left) {
                this._selectionStartXPercent = this._initialSelectionStartXPercent - this._selectionWidthPercent;
            }
            this.drawSelection();
            this.raiseSelectionChanged();
        };
        StackedBarChartView.prototype.drawChart = function () {
            if(!this._viewData) {
                throw new Error(Plugin.Resources.getErrorString("JSPerf.1049"));
            }
            this.clearAll();
            this.createContainer();
            this.drawYAxis();
            this.drawHeader();
            this.drawLegend();
            this.initializeBarGraph();
            this.drawHorizontalScaleLines();
            this.drawVerticalRulerLines();
            this.renderViewData(this._barGraph, this._viewData);
            this._chartAreaContainer.appendChild(this._barGraph);
            this.drawSelection();
        };
        StackedBarChartView.prototype.drawHeader = function () {
            var _this = this;
            this._header = document.createElement("div");
            this._header.style.width = this._options.width + "px";
            this._header.classList.add("stackedBarChartHeader");
            this._collapseExpandArrow = document.createElement("div");
            this._collapseExpandArrow.tabIndex = 0;
            if(this._isCollapsed) {
                this._collapseExpandArrow.classList.add("collapsed");
                this._collapseExpandArrow.setAttribute("aria-label", Plugin.Resources.getString("GraphExpandAriaLabel"));
                this._collapseExpandArrow.setAttribute("aria-expanded", "false");
            } else {
                this._collapseExpandArrow.classList.add("expanded");
                this._collapseExpandArrow.setAttribute("aria-label", Plugin.Resources.getString("GraphCollapseAriaLabel"));
                this._collapseExpandArrow.setAttribute("aria-expanded", "true");
            }
            this._header.appendChild(this._collapseExpandArrow);
            this._collapseExpandArrow.addEventListener("click", function () {
                return _this.toggleExpansion();
            });
            this._collapseExpandArrow.addEventListener("keypress", function (e) {
                if(e.keyCode === KeyCodes.ENTER) {
                    _this.toggleExpansion();
                    _this._collapseExpandArrow.focus();
                }
            });
            this._header.appendChild(this._collapseExpandArrow);
            var description = document.createElement("div");
            description.classList.add("title");
            description.innerText = this._options.title;
            this._header.appendChild(description);
            if(!this._chartAreaContainer) {
                this.rootElement.appendChild(this._header);
            } else {
                this.rootElement.insertBefore(this._header, this._chartAreaContainer);
            }
        };
        StackedBarChartView.prototype.drawHorizontalScaleLines = function () {
            var lineWidth = 1;
            for(var markValue in this._options.yAxisMarks) {
                if(!this._options.yAxisMarks.hasOwnProperty(markValue)) {
                    continue;
                }
                var yPos = Math.ceil(this._options.yAxisMarks[markValue]);
                if(yPos >= this._options.height || yPos <= 0) {
                    continue;
                }
                var line = this.createYAxisLine(yPos);
                this._barGraph.appendChild(line);
            }
        };
        StackedBarChartView.prototype.drawLegend = function () {
            if(!this._options.displayLegend) {
                return;
            }
            var legendDiv = document.createElement("div");
            legendDiv.classList.add("legend");
            for(var seriesName in this._options.legendData) {
                if(!this._options.legendData.hasOwnProperty(seriesName)) {
                    continue;
                }
                var legendItem = this.createLegendItem(this._options.legendData[seriesName].label, this._options.legendData[seriesName].cssClass);
                legendDiv.appendChild(legendItem);
            }
            this._header.appendChild(legendDiv);
        };
        StackedBarChartView.prototype.drawSelection = function () {
            if(!this._leftUnselectedRegion || !this._rightUnselectedRegion) {
                this.initializeUnselectedRegions(this._selectionStartXPercent);
            }
            this._leftUnselectedRegion.style.width = this._selectionStartXPercent + "%";
            this._rightUnselectedRegion.style.left = (this._selectionStartXPercent + this._selectionWidthPercent) + "%";
            this._rightUnselectedRegion.style.width = (100 - (this._selectionStartXPercent + this._selectionWidthPercent)) + "%";
        };
        StackedBarChartView.prototype.drawVerticalRulerLines = function () {
            var positions = this._presenter.verticalRulerLinePositions;
            for(var i = 0; i < positions.length; ++i) {
                var line = this.createVerticalRulerLine(positions[i]);
                this._barGraph.appendChild(line);
            }
        };
        StackedBarChartView.prototype.drawYAxis = function () {
            var yAxisLeft = this.createYAxisDiv("yAxisLeft");
            var yAxisRight = this.createYAxisDiv("yAxisRight");
            for(var markValue in this._options.yAxisMarks) {
                if(!this._options.yAxisMarks.hasOwnProperty(markValue)) {
                    continue;
                }
                this.addYAxisMark(yAxisLeft, markValue, "yAxisMarkLeft");
                this.addYAxisMark(yAxisRight, markValue, "yAxisMarkRight");
            }
        };
        StackedBarChartView.prototype.ensureMinSelectionWidth = function () {
            if(typeof this._selectionStartXPercent === "undefined") {
                return;
            }
            if(Math.abs(this._selectionWidthPercent * this._barGraphWidth / 100) < this._options.minSelectionWidthInPixels) {
                this._selectionWidthPercent = this._options.minSelectionWidthInPixels / this._barGraphWidth * 100;
                switch(this._dragDirection) {
                    case VisualProfiler.DragDirection.right: {
                        if(this._selectionStartXPercent + this._selectionWidthPercent > 100) {
                            this._selectionStartXPercent = 100 - this._selectionWidthPercent;
                        }
                        break;

                    }
                    case VisualProfiler.DragDirection.left: {
                        if(this._selectionStartXPercent - this._selectionWidthPercent < 0) {
                            this._selectionStartXPercent = 0;
                        } else {
                            if(this._selectionStartXPercent + this._selectionWidthPercent > 100) {
                                this._selectionStartXPercent = 100 - this._selectionWidthPercent;
                            } else {
                                this._selectionStartXPercent = this._initialSelectionStartXPercent - this._selectionWidthPercent;
                            }
                        }
                        break;

                    }
                    default: {
                        if(this._selectionStartXPercent + (this._selectionWidthPercent / 2) > 100) {
                            this._selectionStartXPercent = 100 - this._selectionWidthPercent;
                        } else {
                            if(this._selectionStartXPercent - (this._selectionWidthPercent / 2) < 0) {
                                this._selectionStartXPercent = 0;
                            } else {
                                this._selectionStartXPercent = this._selectionStartXPercent - (this._selectionWidthPercent / 2);
                            }
                        }
                        break;

                    }
                }
            }
        };
        StackedBarChartView.prototype.initializeUnselectedRegions = function (startXPercent) {
            this._leftUnselectedRegion = this.createUnselectedRegion(0, startXPercent);
            this._rightUnselectedRegion = this.createUnselectedRegion(startXPercent, 100 - startXPercent);
            this._barGraph.appendChild(this._leftUnselectedRegion);
            this._barGraph.appendChild(this._rightUnselectedRegion);
        };
        StackedBarChartView.prototype.initializeBarGraph = function () {
            var _this = this;
            this._selectedId = -1;
            this._idCount = 0;
            this._barGraph = document.createElement("div");
            this._barGraph.classList.add("barGraph");
            this._barGraph.tabIndex = 0;
            this._barGraph.style.height = this._options.height + "px";
            this._barGraph.style.width = this._barGraphWidth + "px";
            this._barGraph.addEventListener("mousedown", this.startSelection.bind(this));
            this._barGraph.addEventListener("keydown", this.onBarGraphKeydown.bind(this));
            this._barGraph.addEventListener("focus", function () {
                _this._selectedId = -1;
            });
            if(this._options.ariaDescription) {
                this._barGraph.setAttribute("aria-label", this._options.ariaDescription);
            }
            this._dragHandler = this.drag.bind(this);
        };
        StackedBarChartView.prototype.onBarBlur = function (event) {
            var bar = event.currentTarget;
            bar.classList.remove("focused");
            Plugin.VS.Tooltip.dismiss();
        };
        StackedBarChartView.prototype.onBarFocus = function (objectForLabel, event) {
            var bar = event.currentTarget;
            bar.classList.add("focused");
            if(this._options.ariaLabelCallback) {
                var ariaLabel = this._options.ariaLabelCallback(objectForLabel);
                bar.setAttribute("aria-label", ariaLabel);
            }
        };
        StackedBarChartView.prototype.onBarGraphKeydown = function (event) {
            if(event.keyCode === KeyCodes.ARROW_LEFT || event.keyCode === KeyCodes.ARROW_RIGHT) {
                if(event.keyCode === KeyCodes.ARROW_LEFT) {
                    if((this._selectedId === 0) || (this._selectedId === -1)) {
                        this._selectedId = this._idCount;
                    }
                    this._selectedId--;
                } else {
                    if(event.keyCode === KeyCodes.ARROW_RIGHT) {
                        this._selectedId++;
                        if(this._selectedId === this._idCount) {
                            this._selectedId = 0;
                        }
                    }
                }
                var bar = (document.getElementById(StackedBarChartView._barIdPrefix + this._selectedId));
                bar.focus();
                event.preventDefault();
                event.stopPropagation();
                return false;
            }
            return true;
        };
        StackedBarChartView.prototype.onBarKeydown = function (objectForTooltip, event) {
            if(event.keyCode === KeyCodes.ENTER) {
                var element = event.currentTarget;
                var offsetX = window.screenLeft + element.offsetLeft + element.clientWidth;
                var offsetY = window.screenTop + element.offsetTop;
                element = element.offsetParent;
                while(element) {
                    offsetX += element.offsetLeft;
                    offsetY += element.offsetTop;
                    element = element.offsetParent;
                }
                this.showToolTip(objectForTooltip, offsetX, offsetY);
                event.preventDefault();
                event.stopPropagation();
                return false;
            }
            return true;
        };
        StackedBarChartView.prototype.raiseSelectionChanged = function () {
            if(!this._options.selectionCallback) {
                return;
            }
            this._options.selectionCallback(this._selectionStartXPercent, this._selectionWidthPercent);
        };
        StackedBarChartView.prototype.registerPresenterEvents = function () {
            this._presenter.dataSelectionChanged = this.setSelection.bind(this);
        };
        StackedBarChartView.prototype.renderViewData = function (container, viewData) {
            for(var i = 0; i < viewData.length; i++) {
                var barInfo = viewData[i];
                var rectangle = this.createRect(barInfo.x, barInfo.y, barInfo.height, barInfo.width, barInfo.className);
                rectangle.addEventListener("mouseover", this.showToolTip.bind(this, barInfo.attachedObject));
                rectangle.addEventListener("mouseout", Plugin.VS.Tooltip.dismiss);
                rectangle.addEventListener("keydown", this.onBarKeydown.bind(this, barInfo.attachedObject));
                rectangle.addEventListener("focus", this.onBarFocus.bind(this, barInfo.attachedObject));
                rectangle.addEventListener("blur", this.onBarBlur.bind(this));
                container.appendChild(rectangle);
            }
        };
        StackedBarChartView.prototype.setSelection = function (startXPercent, widthPercent) {
            this._selectionStartXPercent = startXPercent;
            this._selectionWidthPercent = widthPercent;
            this.drawSelection();
        };
        StackedBarChartView.prototype.showToolTip = function (objectForTooltip, x, y) {
            if(!this._options.toolTipCallback) {
                return;
            }
            var toolTipContent;
            toolTipContent = this._options.toolTipCallback(objectForTooltip);
            var config = {
                content: toolTipContent,
                delay: 0,
                x: x,
                y: y
            };
            Plugin.VS.Tooltip.show(config);
        };
        StackedBarChartView.prototype.startSelection = function (event) {
            this._initialSelectionStartXPercent = this._selectionStartXPercent = this.convertPageXToChartAreaPercent(event.pageX);
            this._selectionWidthPercent = 0;
            this._dragDirection = StackedBarChartView.getDragDirection(this._selectionWidthPercent);
            this._mouseupHandler = this.stopSelection.bind(this);
            this._chartAreaContainer.addEventListener("mousemove", this._dragHandler);
            this._chartAreaContainer.addEventListener("mouseup", this._mouseupHandler);
            this._chartAreaContainer.setCapture(true);
            this.drawSelection();
        };
        StackedBarChartView.prototype.stopSelection = function (event) {
            this._chartAreaContainer.removeEventListener("mousemove", this._dragHandler);
            this._chartAreaContainer.removeEventListener("mouseup", this._mouseupHandler);
            this._chartAreaContainer.releaseCapture();
            this.ensureMinSelectionWidth();
            this.drawSelection();
            this._dragDirection = VisualProfiler.DragDirection.none;
            this.raiseSelectionChanged();
        };
        StackedBarChartView.prototype.toggleExpansion = function () {
            if(this._collapseExpandArrow.classList.contains("expanded")) {
                this._isCollapsed = true;
                this._chartAreaContainer.style.display = "none";
                this._collapseExpandArrow.classList.remove("expanded");
                this._collapseExpandArrow.classList.add("collapsed");
                this._collapseExpandArrow.setAttribute("aria-label", Plugin.Resources.getString("GraphExpandAriaLabel"));
                this._collapseExpandArrow.setAttribute("aria-expanded", "false");
                Plugin.VS.Internal.CodeMarkers.fire(VisualProfiler.CodeMarkerValues.perfBrowserTools_VisualProfilerGraphCollapsed);
            } else {
                this._isCollapsed = false;
                this.drawChart();
                this._collapseExpandArrow.classList.remove("collapsed");
                this._collapseExpandArrow.classList.add("expanded");
                this._collapseExpandArrow.setAttribute("aria-label", Plugin.Resources.getString("GraphCollapseAriaLabel"));
                this._collapseExpandArrow.setAttribute("aria-expanded", "true");
                Plugin.VS.Internal.CodeMarkers.fire(VisualProfiler.CodeMarkerValues.perfBrowserTools_VisualProfilerGraphExpanded);
            }
        };
        StackedBarChartView.prototype.unregisterPresenterEvents = function () {
            if(!this._presenter) {
                return;
            }
            this._presenter.dataSelectionChanged = null;
        };
        StackedBarChartView.getDragDirection = function getDragDirection(selectionWidth) {
            if(selectionWidth > 0) {
                return VisualProfiler.DragDirection.right;
            } else {
                if(selectionWidth < 0) {
                    return VisualProfiler.DragDirection.left;
                }
            }
            return VisualProfiler.DragDirection.none;
        }
        return StackedBarChartView;
    })(Common.Controls.Control);
    VisualProfiler.StackedBarChartView = StackedBarChartView;    
})(VisualProfiler || (VisualProfiler = {}));
//@ sourceMappingURL=StackedBarChart.js.map

// SIG // Begin signature block
// SIG // MIIamwYJKoZIhvcNAQcCoIIajDCCGogCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFGtoc+2lZMj6
// SIG // s8NcqT1LJa38p1rcoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // jkIAAAAAAB8wDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OFoXDTEz
// SIG // MDQwOTIyMjU1OFowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJbsjkdN
// SIG // VMJclYDXTgs9v5dDw0vjYGcRLwFNDNjRRi8QQN4LpFBS
// SIG // EogLQ3otP+5IbmbHkeYDym7sealqI5vNYp7NaqQ/56ND
// SIG // /2JHobS6RPrfQMGFVH7ooKcsQyObUh8yNfT+mlafjWN3
// SIG // ezCeCjOFchvKSsjMJc3bXREux7CM8Y9DSEcFtXogC+Xz
// SIG // 78G69LPYzTiP+yGqPQpthRfQyueGA8Azg7UlxMxanMTD
// SIG // 2mIlTVMlFGGP+xvg7PdHxoBF5jVTIzZ3yrDdmCs5wHU1
// SIG // D92BTCE9djDFsrBlcylIJ9jC0rCER7t4utV0A97XSxn3
// SIG // U9542ob3YYgmM7RHxqBUiBUrLHUCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBQv6EbIaNNuT7Ig0N6JTvFH7kjB8jAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBz/30unc2NiCt8feNeFXHp
// SIG // aGLwCLZDVsRcSi1o2PlIEZHzEZyF7BLUVKB1qTihWX91
// SIG // 7sb1NNhUpOLQzHyXq5N1MJcHHQRTLDZ/f/FAHgybgOIS
// SIG // CiA6McAHdWfg+jSc7Ij7VxzlWGIgkEUvXUWpyI6zfHJt
// SIG // ECfFS9hvoqgSs201I2f6LNslLbldsR4F50MoPpwFdnfx
// SIG // Jd4FRxlt3kmFodpKSwhGITWodTZMt7MIqt+3K9m+Kmr9
// SIG // 3zUXzD8Mx90Gz06UJGMgCy4krl9DRBJ6XN0326RFs5E6
// SIG // Eld940fGZtPPnEZW9EwHseAMqtX21Tyi4LXU+Bx+BFUQ
// SIG // axj0kc1Rp5VlMIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSOMIIEigIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGwMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBRLhPNQTZEkTuqZmQZCaB4d
// SIG // RPlRSjBQBgorBgEEAYI3AgEMMUIwQKAmgCQAUwB0AGEA
// SIG // YwBrAGUAZABCAGEAcgBDAGgAYQByAHQALgBqAHOhFoAU
// SIG // aHR0cDovL21pY3Jvc29mdC5jb20wDQYJKoZIhvcNAQEB
// SIG // BQAEggEAXOG68YhqR/XIP7DveTFEVdo138b1unMNNlYB
// SIG // 70fmlB023CfuPYWDcgMIiHabAelZjzPGZTrE17ZqmAL9
// SIG // u9IrMZ8+fp1ALU6zPBOaz2RT4oOSszxU0usegqsHtCEg
// SIG // 4zrpzThJBAzaXtRmALZ21GeQKiMHvJ04NJUN41tfq/uV
// SIG // f0/xVy1CcJE4Wjk4u05eGb5ByXSp2apIIuuq7f+HuT9G
// SIG // LXtBEFjSi9Ja88xnz1dk0ExrUKW0aPD95TGBmwMADkS+
// SIG // hvZVuZMuyE37HXgvo+t1zsT2Pu4IlTZdW4hURd+YBNZM
// SIG // CWb9ipK+dupMU3HJROk1CQrOUg5j4UYvgMSuTNLMC6GC
// SIG // Ah8wggIbBgkqhkiG9w0BCQYxggIMMIICCAIBATCBhTB3
// SIG // MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
// SIG // bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
// SIG // cm9zb2Z0IENvcnBvcmF0aW9uMSEwHwYDVQQDExhNaWNy
// SIG // b3NvZnQgVGltZS1TdGFtcCBQQ0ECCmECjkIAAAAAAB8w
// SIG // CQYFKw4DAhoFAKBdMBgGCSqGSIb3DQEJAzELBgkqhkiG
// SIG // 9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTEzMDMxNTA2MzQw
// SIG // MlowIwYJKoZIhvcNAQkEMRYEFD3ztufT9G2M0WWNXjmg
// SIG // i6Z/XqRxMA0GCSqGSIb3DQEBBQUABIIBACtLiDG/h9EQ
// SIG // Hisb2wrNZvQ5tCYS+7lk3rVhYjkBjLQvx/2ItJ+WOHgd
// SIG // yBi9qvvopvXhAOjL4yX+19AW3hbkjj3WYWynanrH6E+i
// SIG // 8VTeeEXRWXWX4WKxoIYJjTWtEb48wmgcdG3HSCArujPW
// SIG // NocrLwfIpM6+ghozfa1XiMnGaFt6QwW1GMr+Hdr3/r3A
// SIG // 3ZqUNxJTBK3LcjklaA4Ai0aIP3x4rs63mxaJKSvtfatC
// SIG // cNW1qmHW8B6Nm02FlaNO8gE6hYVwL8axSnp64aQ3Dn69
// SIG // E86zT1OQ2RShbxLB7SX5YlJJOCE23g6/DR2JR5vog/mE
// SIG // tqH/MqCFpXSJPI0X/cCmzjY=
// SIG // End signature block
